/** @file   redrawqueue.h
 * @brief   Declaration of RedrawQueue - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_REDRAWQUEUE_H
#define H_WWW_REDRAWQUEUE_H

#include <allegro.h>
#include <map>

namespace WeWantWar {

/** @class  RedrawQueue
 * @brief   Handles the correct drawing order.
 * @author  Tomi Lamminsaari
 *
 * The game objects like player and aliens are not drawn directly to the
 * double buffer. They are added this redraw queue which is drawn later.
 * This queue has several "layers" where you can place those sprites.
 * It is then possible to draw sprites on certain layer only. The number
 * of layers is not limited.
 *
 * You can use several different blending modes with the sprites.
 */
class RedrawQueue
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** The types of the bitmap you can add to the queue.
   */
  enum SpType {
    /** Normal bitmap. Drawn with Allegro's @c blit(...) - function */
    SPTYPE_BITMAP,
    /** A sprite. Drawn with Allegro's @c draw_sprite(...) - function */
    SPTYPE_SPRITE,
    /** A RLE-sprite. Drawn with Allegro's @c draw_rle_sprite(...) - function */
    SPTYPE_RLE
  };
  
  /** The blendmodes.
   */
  enum Blendmode {
    /** No blending when drawing the sprite */
    BMODE_NORMAL,
    /** We should use add-blender. */
    BMODE_ADD,
    /** We should use trans-blender. */
    BMODE_TRANS,
    /** We should use multiply-blender. */
    BMODE_MULTIPLY,
    /** We should use burn-blender */
    BMODE_BURN,
    /** We should use dodge-blender */
    BMODE_DODGE,
    /** A predator blender that creates an effect sothing like from the
     * Predator - movies.
     */
    BMODE_PREDATOR
  };


  /** Default layer numbers we use for the objects. This is also the
   * redrawing order.
   */
  enum {
    /** Killed aliens use this layer */
    PRI_KILLED,
    /** Living objects that are small ( like SmallWormAlien ) */
    PRI_BELOW_NORMAL,
    /** Normal priority. Most gameobjects use this. */
    PRI_NORMAL,
    /** Priority the player uses */
    PRI_ABOVE_NORMAL,
    /** Flying objects. Map's Decoration layer can hide objects on this
     * layer.
     */
    PRI_FLYING,
    /** Objects that fly high. Drawn after the decorative map layer. */
    PRI_FLYING_HIGH
  };
  


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	RedrawQueue();


	/** Destructor
   */
	virtual ~RedrawQueue();

private:

	/** Copy constructor.
   * @param     rO                Reference to another RedrawQueue
   */
  RedrawQueue( const RedrawQueue& rO );

	/** Assignment operator
   * @param     rO                Reference to another RedrawQueue
   * @return    Reference to us.
   */
  RedrawQueue& operator = ( const RedrawQueue& rO );

public:


  ///
  /// Methods
  /// =======

  /** Adds new sprite to this queue. 
   * @param     pri               The priority.
   * @param     x                 The x-coordinate of the graphics
   * @param     y                 The y-coordinate of the graphics
   * @param     sptype            Type of the sprite you're adding to this
   *                              queue.
   * @param     pGfx              Pointer to graphics to be drawn. Make sure
   *                              that the actual data at this pointer matches
   *                              with the @c sptype - parameter.
   */
  void add( int pri, int x, int y, SpType sptype, void* pGfx );

  /** Adds new sprite to this queue and sets the blendmode.
   * @param     pri               Priority.
   * @param     x                 X-coordinate
   * @param     y                 Y-coordinate
   * @param     sptype            Type of the sprite
   * @param     blend             Blending mode
   * @param     alpha             Alpha value
   * @param     pGfx              Pointer to graphics
   */
  void add( int pri, int x, int y, SpType sptype, Blendmode blend,
            int alpha, void* pGfx );
            
  /** Adds new rotated sprite to the queue. Rotated sprites cannot be drawn
   * with any other blender mode than @c BMODE_NORMAL. The amount of
   * rotation is a fixed number defined in Allegro.
   * @param     pri               Priority
   * @param     x                 X-coordinate
   * @param     y                 Y-coordinate
   * @param     rot               Rotation. Should be within the range
   *                              0...256
   * @param     pGfx              Pointer to BITMAP
   */
  void addRotatedSprite( int pri, int x, int y, fixed rot, void* pGfx );
            
  /** Adds new polygon to redraw queue
   * @param     pri               Priority of this polygon
   * @param     polytype          The polytype. See Allegro documentation
   * @param     pTex              Texture
   * @param     pV1               Pointer to the 1st vertex
   * @param     pV2               Pointer to the 2nd vertex
   * @param     pV3               Pointer to the 3rd vertex
   */
  void addTriangle( int pri, int polytype, BITMAP* pTex,
                    V3D_f* pV1, V3D_f* pV2, V3D_f* pV3 );
                    
  /** Adds new polygon to the redraw queue
   * @param     pri               Priority of this polygon
   * @param     polytype          The polytype. See Allegro documentation
   * @param     pTex              Texture
   * @param     pV1               Pointer to the 1st vertex
   * @param     pV2               Pointer to the 2nd vertex
   * @param     pV3               Pointer to the 3rd vertex
   * @param     pV4               Pointer to the 4th vertex
   */
  void addQuad( int pri, int polytype, BITMAP* pTex,
                V3D_f* pV1, V3D_f* pV2, V3D_f* pV3, V3D_f* pV4 );

  /** Draws the sprites with given priority.
   * @param     pri               Priority of the sprites being drawn.
   * @param     pTarget           Pointer to target bitmap.
   */
  void redraw( int pri, BITMAP* pTarget );
  
  /** Draws the polygons only.
   * @param     pri               Only polygons with this priority are being
   *                              drawn
   * @param     pTarget           Pointer to target bitmap.
   */
  void drawPolygons( int pri, BITMAP* pTarget );

  /** Clears the queues.
   */
  void clear();
  
  



  ///
  /// Getter methods
  /// ==============
  

protected:

  
  /** The sprites in queue are stored in instances of this structure.
   */
  struct QueueItem {
    /** The x-coordinate the sprite should be drawn. */
    int drawX;
    /** The y-coordinate the sprite should be drawn. */
    int drawY;
    /** Type of this sprite. */
    SpType type;
    /** The blender-mode */
    Blendmode blendmode;
    /** Alpha value */
    int alpha;
    /** Pointer to sprite data itself. */
    void* pGfx;
    /** Rotation */
    fixed rot;
  };
  
  /** The polygons are stored in the instances of ths structure.
   */
  struct PolyItem {
    /** The rendering mode. Use Allegro's POLYTYPE_ - constants. */
    int polytype;
    /** Number of vertices. 3 or 4. */
    int vtxcount;
    /** Pointer to texture bitmap. */
    BITMAP* pTex;
    /** The vertices of the polygon. */
    V3D_f vertices[4];
  };
  
  /** A datatype for creating iterators for the Bitmap - queue */
  typedef std::multimap<int, QueueItem*>::const_iterator BitmapIterator;
  /** A datatype for creating iterators for the Polygon - queue */
  typedef std::multimap<int, PolyItem*>::const_iterator PolyIterator;



  /** Sets the given blender function
   * @param     blend             The blendmode
   * @param     alpha             The alpha value
   */
  void setBlenderFunction( Blendmode blend, int alpha );




  ///
  /// Members
  /// =======

  
  /** This vector holds the sprites that are added to the queue */
  std::multimap< int, QueueItem* > m_bitmapQueue;
  
  /** This vector is the polygon queue */
  std::multimap< int, PolyItem* > m_polyQueue;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: redrawqueue.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:12+02  lamminsa
 * Initial revision
 *
 */
 
